"""
tick_state.py

Defines the TickState dataclass storing the context distribution and radius N.

Attributes
----------
distribution : np.ndarray
    1D array of length 2*N+1 representing tick‑mass or amplitude across contexts.
N : int
    Half‑range (context depth), so total lattice size is 2*N+1.
"""

from dataclasses import dataclass
import numpy as np


@dataclass(frozen=True)
class TickState:
    distribution: np.ndarray
    N: int

    def __post_init__(self):
        if not isinstance(self.distribution, np.ndarray):
            raise TypeError(
                f"distribution must be a numpy.ndarray, got {type(self.distribution)}"
            )
        if self.distribution.ndim != 1:
            raise ValueError(
                f"distribution must be a 1D array, got shape {self.distribution.shape}"
            )
        expected_len = 2 * self.N + 1
        if self.distribution.size != expected_len:
            raise ValueError(
                f"distribution length must be 2*N+1 ({expected_len}), got {self.distribution.size}"
            )

    def __repr__(self):
        return f"TickState(distribution=array(shape={self.distribution.shape}), N={self.N})"
